/**
 * @license
 *
 * Copyright 2025 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { genkitPlugin } from "genkit/plugin";
import { listModels } from "./client.js";
import * as embedder from "./embedder.js";
import * as gemini from "./gemini.js";
import * as imagen from "./imagen.js";
import * as lyria from "./lyria.js";
import * as veo from "./veo.js";
import { getDerivedOptions } from "./utils.js";
async function initializer(ai, pluginOptions) {
  const clientOptions = await getDerivedOptions(pluginOptions);
  veo.defineKnownModels(ai, clientOptions, pluginOptions);
  imagen.defineKnownModels(ai, clientOptions, pluginOptions);
  lyria.defineKnownModels(ai, clientOptions, pluginOptions);
  gemini.defineKnownModels(ai, clientOptions, pluginOptions);
  embedder.defineKnownModels(ai, clientOptions, pluginOptions);
}
async function resolver(ai, actionType, actionName, pluginOptions) {
  const clientOptions = await getDerivedOptions(pluginOptions);
  switch (actionType) {
    case "model":
      if (lyria.isLyriaModelName(actionName)) {
        lyria.defineModel(ai, actionName, clientOptions, pluginOptions);
      } else if (imagen.isImagenModelName(actionName)) {
        imagen.defineModel(ai, actionName, clientOptions, pluginOptions);
      } else if (veo.isVeoModelName(actionName)) {
      } else {
        gemini.defineModel(ai, actionName, clientOptions, pluginOptions);
      }
      break;
    case "background-model":
      if (veo.isVeoModelName(actionName)) {
        veo.defineModel(ai, actionName, clientOptions, pluginOptions);
      }
      break;
    case "embedder":
      embedder.defineEmbedder(ai, actionName, clientOptions, pluginOptions);
      break;
    default:
  }
}
async function listActions(options) {
  try {
    const clientOptions = await getDerivedOptions(options);
    const models = await listModels(clientOptions);
    return [
      ...gemini.listActions(models),
      ...imagen.listActions(models),
      ...lyria.listActions(models),
      ...veo.listActions(models)
      // We don't list embedders here
    ];
  } catch (e) {
    return [];
  }
}
function vertexAIPlugin(options) {
  let listActionsCache;
  return genkitPlugin(
    "vertexai",
    async (ai) => await initializer(ai, options),
    async (ai, actionType, actionName) => await resolver(ai, actionType, actionName, options),
    async () => {
      if (listActionsCache) return listActionsCache;
      listActionsCache = await listActions(options);
      return listActionsCache;
    }
  );
}
const vertexAI = vertexAIPlugin;
vertexAI.model = (name, config) => {
  if (imagen.isImagenModelName(name)) {
    return imagen.model(name, config);
  }
  if (lyria.isLyriaModelName(name)) {
    return lyria.model(name, config);
  }
  if (veo.isVeoModelName(name)) {
    return veo.model(name, config);
  }
  return gemini.model(name, config);
};
vertexAI.embedder = (name, config) => {
  return embedder.model(name, config);
};
var vertexai_default = vertexAI;
export {
  vertexai_default as default,
  vertexAI
};
//# sourceMappingURL=index.mjs.map